;File:	RASM1.ASM
;Edit date:	93/01/14.
;Serial number 12
;
;	RP/M2 assembler I/O module.
;	With sorted cross reference modifications.
;
IOMORG	EQU	0300H	;module fwa
;
;	Locations on page 0.
;
RWBT	EQU	0000H	;warmboot vector
RDOS	EQU	0005H	;RDOS vector
RFCB	EQU	005CH	;resident fcb
RBUF	EQU	0080H	;resident buffer
;
;	RDOS functions.
;
FCCI	EQU	 1	;console character in
FCCO	EQU	 2	;console character out
FRCI	EQU	 3	;reader character in
FCST	EQU	11	;console input status
FSEL	EQU	14	;select disk
FOPN	EQU	15	;open file
FCLO	EQU	16	;close file
FDEL	EQU	19	;delete file
FRSR	EQU	20	;read sequential record
FWSR	EQU	21	;write sequential record
FCNF	EQU	22	;create new file
FRCD	EQU	25	;return current disk
FDMA	EQU	26	;set disk record address
FRRR	EQU	33	;read random record
FWRR	EQU	34	;write random record
FSRR	EQU	36	;set random record
;
;	Locations on page 1.
;	Print line buffer.
;
PLBFWA	EQU	010CH	;line buffer fwa
PLBSIZ	EQU	120	;line length
PLBFBP	EQU	PLBFWA+PLBSIZ	;buffer fill pointer
;
;	Assembler control data.
;
TOKEN	EQU	PLBFBP+1	;current token
VALUE	EQU	TOKEN+1		;binary value
ACCLEN	EQU	VALUE+2		;accumulator length
ACCUM	EQU	ACCLEN+1	;accumulator fwa
ACCSIZ	EQU	64
EVALUE	EQU	ACCUM+ACCSIZ	;expression value
SYTOP	EQU	EVALUE+2	;current symbol table top
SYMAX	EQU	SYTOP+2		;symbol table lwa + 1
PASSN	EQU	SYMAX+2		;pass number, 0 or 1
HEXPC	EQU	PASSN+1		;current hex fill address
LOCCN	EQU	HEXPC+2		;assembler's location counter
SYBAS	EQU	LOCCN+2		;symbol table base
SYADR	EQU	SYBAS+2		;current symbol address
SYMORD	EQU	SYADR+2		;symbol entry ordinal
;
;	Ascii character codes.
;
cr	EQU	0DH		;carriage return
lf	EQU	0AH		;line feed
eof	EQU	1AH		;control-z = end of file
;
;	File buffer definitions.
;
NSFBF	EQU	16		;source buffer record capacity
NPFBF	EQU	12		;print buffer record capacity
NHFBF	EQU	12		;hex buffer record capacity
;
SSIZE	EQU	NSFBF*128	;source buffer byte size
PSIZE	EQU	NPFBF*128	;print buffer byte size
HSIZE	EQU	NHFBF*128	;hex buffer byte size
;
;	I/O module entry points.
;
	ORG	IOMORG
;
	JMP	PRS	;preset assembler
	JMP	RIF	;reset input file
	JMP	GNC	;get next source character
	JMP	PNC	;put next print character
	JMP	PNB	;put next hex character
	JMP	CCO	;console character out
	JMP	MSG	;console message out
	JMP	WPL	;write print line
	JMP	PER	;pack error flag
	JMP	PHB	;pack hex byte
	JMP	EOR	;process end of assembly
	JMP	PCF	;process source file chaining
	JMP	RFN	;restore original source filename
	JMP	PIF	;preset the include file
	JMP	CSF	;switch source to include file
	JMP	BFB	;blank fill print line buffer
	JMP	MXR	;create cross reference record
	JMP	AXR	;advance cross reference record
	JMP	BRK	;process console break
;
;	Data space.
;
HEXBPC:	DW	0	;current hex PC
HEXRCL:	DB	0	;current hex byte count
HEXBUF:	DS	16	;hex record buffer
;
;	Variables for XREF sort.
;
TSLWA:	DS	2	;symbol address table lwa+1
TSSIZ:	DS	2	;symbol count
NDXHS:	DS	2	;saved index high
NDXHI:	DS	2	;index high
NDXLO:	DS	2	;index low
SWAPF:	DS	1	;swap flag
;
;	Disk names.
;
CDISK:	DB	0	;current disk
ADISK:	DB	0	;source file disk
PDISK:	DB	0	;print file disk
HDISK:	DB	0	;hex file disk
XDISK:	DB	0	;cross reference file disk
;
;	File control blocks.
;
SFCB:	DB	0,'        ASM',0,0,0,0
SFEX	EQU	$-4	;extent
	DB	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
SFCR:	DB	0,0,0,0
;
;	Print file.
;
PFCB:	DB	0,'        PRN',0,0,0,0
	DB	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
	DB	0,0,0,0
;
;	Hex file.
;
HFCB:	DB	0,'        HEX',0,0,0,0
	DB	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
	DB	0,0,0,0
;
;	Cross reference file.
;
XFCB:	DB	0,'$XREF   $XR',0,0,0,0
	DB	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
XFCBCR	EQU	$
	DB	0,0,0,0
XRFRRN	EQU	$-3	;random record number
;
;	Library include file.
;
IFLAG:	DB	0	;1=include active
IFCB:	DB	0,'INCLUDE LIB',0,0,0,0
	DB	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
	DB	0,0,0,0
;
;	Source file buffer.
;
SFSBP:	DW	0		;save source buffer pointer
SFBP:	DW	SSIZE		;source file buffer pointer
SBUFF:	DS	SSIZE
;
;	Print file buffer.
;
PFBP:	DW	0		;print file buffer pointer
PBUFF:	DS	PSIZE
;
;	Hex file buffer.
;
HFBP:	DW	0		;hex file buffer pointer
HBUFF:	DS	HSIZE
HRTYPE:	DB	0		;hex record type
;
;	Cross reference record prototype.
;
XRFREC	EQU	$
	DW	0001H
XRFSYM:	DW	0000H		;symbol table entry address
XRFLOC:	DW	0000H		;first reference to this symbol
;
;	Cross reference record buffer.
;
XRFBUF	EQU	$
	DS	128
;
;	SEL - Select disk.
;	Entry	 A = drive number
;
SEL:	LXI	H,CDISK
	CMP	M
	RZ		;If we are there
;
	MOV	M,A	;set CDISK
	MOV	E,A
	MVI	C,FSEL	;select CDISK
	CALL	RDOS
	RET
;
;	SDN - Scan disk name parameters.
;	Entry	HL = current scan location
;	Exit	HL = next scan location
;		 A = drive name
;
SDN:	MOV	A,M
	INX	H
	CPI	' '
	JZ	SDN1	;If default drive indicated
;
	SBI	'A'
	RET
;
SDN1:	LDA	CDISK
	RET
;
;	MMC - Move memory bytes.
;	Entry	HL = source fwa
;		DE = destination fwa
;		 C = byte count
;
MMC:	MOV	A,M
	STAX	D
	INX D ! INX H
	DCR	C
	JNZ	MMC	;loop for C bytes
	RET
;
;	WSR - Write sequential record.
;	Entry	DE = .fcb
;	Exit	 Z = false, if disk is full
;
WSR:	MVI	C,FWSR
	CALL	RDOS
	ORA	A
	RET
;
;	PEL - Issue cr,lf to print file.
;
PEL:	MVI	A,cr
	CALL	WPC
	MVI	A,lf
	CALL	WPC
	RET
;
;	BFB - Blank fill the print line buffer.
;
BFB:	LXI	H,PLBFWA
	MVI	A,PLBSIZ	;buffer size
BFB1:	MVI	M,' '
	INX	H
	DCR	A
	JNZ	BFB1	;loop over line buffer
	RET
;
;	RCI - Reader character in.
;	Exit	 A = COM1 status
;		DE = fwa common data area
;
RCI:	MVI	C,FRCI
	CALL	RDOS
	RET
;
;	CCI - Console character in.
;	Exit	 A = char
;
CCI:	MVI	C,FCCI
	CALL	RDOS
	RET
;
;	CST - Get console input status.
;	Exit	 Z = false, if key pressed
;
CST:	MVI	C,FCST
	CALL	RDOS
	ORA	A
	RET
;
;	BRK - Process console break.
;
BRK:	CALL	CST	;check keyboard status
	RZ		;If no key pressed
;
;	Console break.
;
	LXI	D,BRKA	;"stop?"
	CALL	MSG
	CALL	CCI	;get response
	CALL	CCI
	ANI	5FH
	CPI	'Y'
	JZ	EOR4	;If stop
;
	RET
;
BRKA:	DB	cr,lf,'Stop? (y/n): ',0
;
;	CCO - Process console character out.
;	Entry	 A = character
;
CCO:	PUSH B ! PUSH D ! PUSH H
	MVI	C,FCCO
	MOV	E,A
	CALL	RDOS
	POP H ! POP D ! POP B
	RET
;
;	MSG - Console message out.
;	Message ends in 00.
;	Entry	DE = message fwa
;
MSG:	LDAX	D
	CPI	00
	RZ		;If end of message
;
	CALL	CCO
	INX	D
	JMP	MSG
;
;	GFN - Get source filename.
;	Entry	HL = target fwa
;
GFN:	LXI	D,RFCB+1	;resident fcb
	INX	H
	MVI	B,8
GFN1:	LDAX	D
	CPI	'?'
	JZ	GFN2	;If ambiguous
;
	MOV	M,A
	INX H ! INX D
	DCR	B
	JNZ	GFN1	;loop over filename
	RET
;
;	Process ambiguous source filename.
;
GFN2:	LXI	D,GFNA	;"use unambiguous filename"
	CALL	MSG
	JMP	RWBT
;
GFNA:	DB	cr,lf,'Source file cannot be ambiguous.',cr,lf,0
;
;	RFN - Restore the source filename.
;
RFN:	LXI	H,SFCB	;restore original source file
	CALL	GFN
	LXI	D,RFNA	;restore filetype=ASM
	MVI	B,3
RFN1:	LDAX	D
	MOV	M,A
	INX	H
	INX	D
	DCR	B
	JNZ	RFN1	;loop over the filename
	RET
;
RFNA:	DB	'ASM'
;
;	RCD - Return current disk.
;	Exit	 A = current drive number
;
RCD:	MVI	C,FRCD
	CALL	RDOS
	RET
;
;	OPN - Open file.
;	Entry	DE = .fcb
;	Exit	 Z = true, if file not found
;
OPN:	MVI	C,FOPN
	CALL	RDOS
	CPI	0FFH
	RET
;
;	CLO - Close file.
;	Entry	DE = .fcb
;	Exit	 Z = true, if close error
;
CLO:	MVI	C,FCLO
	CALL	RDOS
	CPI	0FFH
	RET
;
;	RSR - Read sequential record.
;	Entry	DE = .fcb
;	Exit	 Z = true, if not at eoi
;
RSR:	MVI	C,FRSR
	CALL	RDOS
	ORA	A
	RET
;
;	SRR - Set random record.
;	Entry	DE = .fcb
;
SRR:	MVI	C,FSRR
	CALL	RDOS
	RET
;
;	RRR - Read random record.
;	Entry	DE = .fcb
;	Exit	 Z = true, if not eoi
;
RRR:	MVI	C,FRRR
RRR1:	CALL	RDOS
	ORA	A
	RET
;
;	WRR - Write random record.
;	Entry	DE = .fcb
;
WRR:	MVI	C,FWRR
	JMP	RRR1
;
;	SBA - Set default buffer address.
;	Exit	DMA set to RBUF
;
SBA:	LXI	H,RBUF
;
;	SDA - Set disk buffer address.
;	Entry	HL = DMA address
;
SDA:	PUSH	H
	XCHG
	MVI	C,FDMA
	CALL	RDOS
	POP	H
	RET
;
;	NPR - Check print parameter.
;	Exit	 Z = true, if no print file
;
NPR:	LDA	PDISK
	CPI	'Z'-'A'
	RZ		;If no print file
;
	CPI	'X'-'A'	;check print to console
	RET
;
;	SPF - Select print file drive.
;
SPF:	LDA	PDISK
	CALL	SEL
	RET
;
;	SHF - Select hex file drive.
;
SHF:	LDA	HDISK
	CALL	SEL
	RET
;
;	SSF - Select source file drive.
;
SSF:	LDA	ADISK
	CALL	SEL
	RET
;
;	DEL - Delete file.
;	Entry	DE = .fcb
;
DEL:	MVI	C,FDEL
	CALL	RDOS
	RET
;
;	CNF - Create new file.
;	Entry	DE = .fcb
;	Exit	 Z = true, if no directory space
;
CNF:	MVI	C,FCNF
	CALL	RDOS
	CPI	0FFH
	RET
;
;	MXF - Create the cross reference file.
;
MXF:	LXI	H,XFCB	;copy filename
	CALL	GFN
	LXI	D,XFCB
	CALL	DEL
	LXI	D,XFCB
	CALL	CNF
	RET
;
;	HLA - HL = HL + A.
;
HLA:	ADD L ! MOV L,A ! RNC	;If done
	INR H ! RET
;
;	CHD - Compare HL:DE.
;		HL=DE	HL<>DE	HL>=DE	HL<DE
;	  Z	  1	  0	  x	  0
;	  C	  0	  x	  0	  1
;
CHD:	MOV A,H ! SUB D ! RNZ	;If different
	MOV A,L ! SUB E ! RET
;
;	HSR - HL shift right.
;
HSR:	ORA	A
	MOV A,H ! RAR ! MOV H,A
	MOV A,L ! RAR ! MOV L,A
	RET
;
;	MUL - Multiply.
;	Set HL = HL * DE.
;
MUL:	MOV B,H ! MOV C,L
	LXI	H,0
MUL1:	XRA	A	;clear carry
	MOV A,B ! RAR ! MOV B,A
	MOV A,C ! RAR ! MOV C,A
	JC	MUL2	;If lsb=1
;
	ORA B ! RZ	;If done
	JMP	MUL3
;
MUL2:	DAD	D
MUL3:	XCHG		;shift HL left
	DAD	H
	XCHG
	JMP	MUL1
;
;	ASA - Advance symbol address.
;	Entry	HL = current symbol table address
;	Exit	HL = next symbol table address
;		 Z = true, if at end of symbols
;	Stores symbol sort value in the collision field.
;
ASA:	PUSH B ! PUSH D
	PUSH H
	CALL	CSV	;HL=symbol sort value
	XCHG
	POP	H
	MOV M,E ! INX H ! MOV M,D ! INX H
	MOV	A,M	;get symbol length-1
	ANI	0FH
	ADI	6	;point to next symbol
	CALL	HLA
	XCHG		;check end of symbols
	LHLD	SYTOP
	XCHG
	CALL	CHD	;compare symbol address:sym table top
	POP D ! POP B
	RET
;
;	TSA - Build a table of symbol addresses.
;	This is the table that will be sorted into
;	alphabetic order of symbol names.
;	Exit	HL = table lwa+1
;		BC = number of entries in the table
;
TSA:	LXI	H,ENDIOM	;use code space for this table
	XCHG
	LHLD	SYBAS		;address of the first symbol
	LXI	B,0000		;reset symbol count
TSA1:	INX	B		;advance symbol count
	XCHG
	MOV M,E ! INX H ! MOV M,D ! INX H
	XCHG
	CALL	ASA		;advance to next symbol
	JNZ	TSA1		;loop to end of symbol table
;
	XCHG			;HL=lwa+1
	RET
;
;	GRN - Get random record number from symbol table.
;	Entry	HL = symbol table entry fwa
;	Exit	HL = record number
;
GRN:	INX H ! INX H		;skip collision field
	MOV	A,M		;get length-1
	ANI	0FH
	INR	A
	INX	H
	CALL	HLA		;point to symbol value field
	INX H ! INX H		;point to the record number
	MOV E,M ! INX H ! MOV D,M
	XCHG
	RET
;
;	CCA - Convert character to alpha.
;	Entry	 A = char, may be Ascii numeric
;	Exit	 A = char, Ascii alphabetic
;
CCA:	CPI	'A'
	RNC			;If alpha
;
;	Convert '0' to 'A', etc.
;
	ADI	'A'-'0'
	RET
;
;	GSC - Get next symbol character.
;	Entry	HL = current symbol table location
;		 C = number of symbol characters remaining
;	Exit	HL = next symbol table location
;		 C = remaining character count
;		 A = case folded character - '@'
;
GSC:	MOV	A,C
	ORA	A
	JNZ	GSC1		;If not past end of symbol string
;
;	We are past the end of the symbol string.
;	Return a space character.
;
	MVI	A,' '
	JMP	GSC2
;
;	Get, and adjust for sorting, the next symbol character.
;
GSC1:	MOV	A,M
	CALL	CCA		;convert to alphabetic
	DCR	C
	ANI	5FH		;fold to upper case
GSC2:	SUI	'@'		;adjust for sorting
	INX	H
	RET
;
;	CSV - Calculate symbol sort value.
;	Sort value = ((a1-'@')*32 + (a2-'@'))*32 + (a3-'@')
;	where a1,a2,a3 are the first three characters of the
;	symbol printname.
;	Entry	HL = symbol table entry fwa
;	Exit	HL = symbol sort value
;
CSV:	INX H ! INX H		;skip collision field
	MOV	A,M		;get length-1
	ANI	0FH
	INR	A		;symbol length
	MOV	C,A
	INX	H
	CALL	GSC		;get a1-bias
	PUSH B ! PUSH H
	MOV E,A ! MVI D,0	;DE=(a1-'@')
	LXI	H,32
	CALL	MUL		;HL=(a1-'@')*32
	XCHG
	POP H ! POP B
	CALL	GSC		;get a2-bias
	XCHG
	CALL	HLA		;HL=(a1-'@')*32 + (a2-'@')
	XCHG
	PUSH B ! PUSH H
	LXI	H,32
	CALL	MUL		;HL=((a1-'@')*32 + (a2-'@'))*32
	XCHG
	POP H ! POP B
	CALL	GSC		;get a3-bias
	XCHG
	CALL	HLA		;HL=((a1-'@')*32 + (a2-'@'))*32 + (a3-'@')
	RET
;
;	GTA - Get symbol address table entry address.
;	Entry	HL = table ordinal
;	Exit	HL = entry address
;
GTA:	PUSH	D
	DAD	H		;*2 for word table
	XCHG
	LXI	H,ENDIOM	;table fwa
	DAD	D
	POP	D
	RET
;
;	GSA - Get symbol table entry address.
;	Entry	HL = address table entry address
;	Exit	HL = symbol table entry address
;
GSA:	PUSH	D
	MOV E,M ! INX H ! MOV D,M
	XCHG
	POP	D
	RET
;
;	SWP - Check symbol address table swap required.
;	Entry	HL = low symbol value
;		DE = high symbol value
;
SWP:	CALL	CHD		;compare low:high
	MVI	A,0		;show no swap
	RC ! RZ			;If no swap required
;
;	Swap these two address table entries.
;
	LHLD	NDXLO		;get low entry
	CALL	GTA		;HL=low address table address
	PUSH	H
	LHLD	NDXHI		;get high entry
	CALL	GTA		;HL=high address table address
	POP	D
	MOV C,M ! LDAX D
	MOV M,A ! MOV A,C ! STAX D
	INX H ! INX D
	MOV C,M ! LDAX D
	MOV M,A ! MOV A,C ! STAX D
	MVI	A,1		;show swap done
	RET
;
;	RXH - Get high sort value.
;	Exit	HL = sort value at high index
;
RXH:	PUSH B ! PUSH D
	LHLD	NDXHI
	CALL	GTA		;HL=address table entry address
	CALL	GSA		;HL=symbol table entry address
	MOV E,M ! INX H ! MOV D,M
	XCHG
	POP D ! POP B
	RET
;
;	RXL - Get low sort value.
;	Exit	HL = sort value at low index
;
RXL:	PUSH B ! PUSH D
	LHLD	NDXLO
	CALL	GTA		;HL=address table entry address
	CALL	GSA		;HL=symbol table entry address
	MOV E,M ! INX H ! MOV D,M
	XCHG
	POP D ! POP B
	RET
;
;	SXR - Sort the cross reference symbol address table.
;
SXR:	LHLD	TSSIZ		;total number of symbols
	DCX	H
	MOV A,H ! ORA L ! RZ	;If no sort required
;
	INX H ! JMP SXR4
;
;	Reset indexes.
;
SXR1:	SHLD	NDXHI		;reset index high
	LXI	H,0000		;reset index low
	SHLD	NDXLO
	XRA	A		;reset swap flag
	STA	SWAPF
SXR2:	CALL	RXH		;HL=high sort value
	PUSH	H
	CALL	RXL		;HL=low sort value
	POP	D
	CALL	SWP		;check and swap
	ORA	A
	JZ	SXR3		;If no swap done
;
;	Mark swap done.
;
	STA	SWAPF
;
;	Advance indexes.
;
SXR3:	LHLD	NDXLO
	INX	H
	SHLD	NDXLO
	LHLD	NDXHI
	INX	H
	SHLD	NDXHI
	XCHG
	LHLD	TSSIZ		;check end of pass
	CALL	CHD
	JNZ	SXR2		;loop to end of address table
;
;	If no swap was done, shorten the sort range.
;
	LDA	SWAPF
	ORA	A
	LHLD	NDXHS
	JNZ	SXR1		;If a swap was done
;
;	Adjust sort range.
;
	DCX	H
	MOV A,H ! ORA L
	INX	H
	JZ	SXR5		;If we are at range = 1
;
;	Reduce sort range.
;
SXR4:	CALL	HSR		;range=range/2
	SHLD	NDXHS		;save the sort range
	JMP	SXR1		;loop while range greater than 1
;
;	We are at range = 1.
;
SXR5:	LDA	SWAPF		;check swap done
	ORA	A
	JNZ	SXR1		;loop until no swaps occur
;
	RET
;
;	ERF - Issue filename error.
;
ERF:	PUSH B ! PUSH D ! PUSH H
	MVI A,'F' ! CALL PER
	POP H ! POP D ! POP B
	RET
;
;	PRS - Program preset.
;
PRS:	LXI	D,PRSA	;"Assembler version number"
	CALL	MSG
	LDA	RFCB+1
	CPI	' '
	JZ	PRS3	;If no source file stated
;
	CALL	RCD	;get A=current disk number
	STA	CDISK
	INR	A
	STA	IFCB
	LXI	H,RFCB+1+8	;get drive names
	CALL SDN ! STA ADISK	;source disk
	CALL SDN ! STA HDISK	;hex disk
	CALL SDN ! STA PDISK	;print disk
;
;	Preset autoselect for the include file.
;
	LDA	RFCB+16
	ORA	A
	JZ	PRS0		;If l: not stated
;
	STA	IFCB
;
;	Preset autoselect the cross reference file.
;
PRS0:	LDA	RFCB
	STA	XFCB
	ORA	A
	CNZ	MXF	;If cross reference requested
;
;	Copy the filename.
;
	LXI	H,SFCB	;copy to source fcb
	CALL	GFN
	CALL	NPR
	JZ	PRS1	;If no print file
;
;	Set up the print file.
;
	LXI	H,PFCB	;copy filename to print fcb
	PUSH H ! PUSH H
	CALL	GFN
	CALL	SPF	;select PDISK
	POP	D	;delete any previous .PRN
	CALL	DEL
	POP	D	;create new print file
	CALL	CNF
	JZ	PRS4	;If directory is full
;
;	Set up the object code hex file.
;
PRS1:	LDA	HDISK
	CPI	'Z'-'A'
	JZ	PRS2	;If no object file
;
;	Set up the hex file.
;
	LXI	H,HFCB	;copy filename to hex fcb
	PUSH H ! PUSH H
	CALL	GFN
	CALL	SHF	;select HDISK
	POP	D	;delete any previous .HEX
	CALL	DEL
	POP	D	;create new hex file
	CALL	CNF
	JZ	PRS4	;If directory is full
;
;	Go to the assembler.
;
PRS2:	JMP	ENDIOM
;
;	Process source file not stated.
;
PRS3:	LXI	D,PRSB	;"no source file"
	CALL	MSG
	JMP	RWBT
;
;	Process directory full.
;
PRS4:	LXI	D,PRSC	;"directory full"
	CALL	MSG
	JMP	RWBT
;
PRSA:	DB	cr,'PC RP/M2 Assembler v1.4 (c) 1986 microMethods.',cr,lf,0
PRSB:	DB	cr,lf,'No source file stated.',cr,lf,0
PRSC:	DB	cr,lf,'Directory is full.',cr,lf,0
;
;	MXR - Create a cross reference record.
;	Entry	DE = SYMORD
;	Exit	DE = SYMORD
;
MXR:	LDA	XFCB
	ORA	A
	RZ		;If no cross reference
;
	LDA	PASSN
	ORA	A
	RNZ		;If not first pass
;
	PUSH	D
	XCHG
	SHLD	XRFRRN	;set record number
	LHLD	SYADR	;set symbol table entry address
	SHLD	XRFSYM
	LHLD	LOCCN	;set l1
	SHLD	XRFLOC
	LXI	H,XRFREC	;set DMA
	CALL	SDA
	LXI	D,XFCB	;write record
	CALL	WRR
	POP	D
	RET
;
;	AXR - Advance cross reference record.
;	Entry	DE = SYMORD
;
AXR:	LDA	PASSN
	CPI	1
	RNZ		;If not second pass
;
	XCHG
	SHLD	XRFRRN	;set record number
	XCHG
	LDA	XFCB
	ORA	A
	RZ		;If no cross reference
;
	LXI	H,XRFBUF	;set DMA
	CALL	SDA
	LXI	D,XFCB	;read the record
	CALL	RRR
;
	LXI	H,XRFBUF	;advance n
	MOV	A,M
	INR	A
	ANI	3FH
	RZ		;If cross ref capacity exceeded
;
	NOP ! NOP
	MOV	M,A
	ADD	A
	CALL	HLA	;set entry(n)=LOCCN
	XCHG
	LHLD	LOCCN
	XCHG
	MOV	M,E
	INX	H
	MOV	M,D
	LXI	D,XFCB	;write the record
	CALL	WRR
	RET
;
;	RIF - Reset the source input file.
;
RIF:	LXI	H,SSIZE	;mark buffer empty
	SHLD	SFBP
	XRA	A	;reset source fcb
	STA	SFEX
	STA	SFCR
	STA	HEXRCL	;reset hex fill buffer
	CALL	SSF	;select ADISK
	LXI	D,SFCB	;open the source file
	CALL	OPN
	RNZ		;If file found
;
	LXI	D,RIFA	;"file not found"
	CALL	MSG
	JMP	RWBT
;
RIFA:	DB	cr,lf,'Source file not found.',cr,lf,0
;
;	GAC - Get next accumulator character.
;	Entry	 C = char count
;	Exit	 C = char count
;		 Z = true, if no data
;
GAC:	PUSH	H
	LXI	H,ACCLEN
	MOV	A,M
	INR	M	;advance char index
	INX	H
	CALL	HLA
	MOV	A,M	;get char
	POP	H
	DCR	C	;advance countdown
	RET
;
;	CFN - Copy filename from accumulator.
;	Entry	DE = destination fwa
;
CFN:	LXI	H,ACCLEN	;get char count
	MOV	C,M
	INR	C
	MVI	M,0	;reset char index
	MVI	B,8
CFN1:	CALL	GAC	;get next char
	CZ	ERF	;If accumulator empty
;
	CPI	'.'
	JZ	CFN2	;If end of filename
;
	STAX	D
	INX	D
	DCR	B
	JZ	CFN3	;If 8 characters moved
;
	JMP	CFN1
;
;	Blank fill short filename.
;
CFN2:	MVI	A,' '
	STAX	D
	INX	D
	DCR	B
	JNZ	CFN2	;loop to end of filename
;
;	Process filetype.
;
CFN3:	MVI	B,3
CFN4:	CALL	GAC	;get next char
	JZ	CFN5	;If end of filetype
;
	STAX	D
	INX	D
	DCR	B
	JNZ	CFN4	;loop over filetype
;
	RET
;
;	Blank fill short filetype.
;
CFN5:	MVI	A,' '
	STAX	D
	INX	D
	DCR	B
	JNZ	CFN5	;loop over filetype
;
	RET
;
;	CSF - Switch source to include file.
;	Entry	HL = .switch request flag
;
CSF:	MOV	A,M
	ORA	A
	RZ		;If no switch request
;
;	Change our character source to the include file.
;
	STA	IFLAG
	XRA	A
	MOV	M,A	;clear switch request
	LHLD	SFBP	;mark our source buffer position
	SHLD	SFSBP
	CALL	RIF	;reset source buffer
	RET
;
;	PIF - Preset the include file.
;	Mark our source file positon.
;
PIF:	LXI	D,IFCB+1	;get the include filename
	CALL	CFN	;copy library filename
	LXI	D,IFCB	;open the include file
	CALL	OPN
	JZ	PIF1	;If file not found
;
	MVI	A,0	;reset current record
	STA	IFCB+32
	RET
;
;	Process include file not found.
;
PIF1:	MVI	A,0	;display the filename
	STA	IFCB+1+8+3
	LXI	D,IFCB+1
	CALL	MSG
	LXI	D,PIFA	;"not found"
	CALL	MSG
	JMP	RWBT
;
PIFA:	DB	' not found.',cr,lf,0
;
;	GIC - Get next include character.
;	Exit	 A = character
;
GIC:	PUSH B ! PUSH D ! PUSH H
	LHLD	SFBP
	LXI	D,SSIZE
	CALL	CHD
	JNZ	GIC2	;If data available
;
;	Reload the source buffer.
;
	LXI	H,0	;reset the source pointer
	SHLD	SFBP
	MVI	B,NSFBF	;records to read
	LXI	H,SBUFF	;buffer fwa
;
;	Read next record.
;
GIC1:	PUSH B ! PUSH H
	CALL	SDA	;set DMA
	LXI	D,IFCB
	CALL	RSR
	POP H ! POP B
	MVI	C,128
	JNZ	GIC3	;If end of information
;
;	Advance buffer.
;
	MOV	A,C	;advance DMA
	CALL	HLA
	DCR	B
	JNZ	GIC1	;loop for B records
;
;	Get next character.
;
GIC2:	LXI	D,SBUFF
	LHLD	SFBP
	PUSH	H	;save current pointer
	INX	H	;advance buffer pointer
	SHLD	SFBP
	POP	H
	DAD	D
	MOV	A,M
	CPI	eof
	CZ	PEI	;If end of include file
;
	POP H ! POP D ! POP B
	RET
;
;	Process end of information on include file.
;
GIC3:	MVI	M,eof	;pack record with eof marks
	INX	H
	DCR	C
	JNZ	GIC3	;loop to end of record
	JMP	GIC2
;
;	PEI - Process end of include file.
;	Restore the source file.
;
PEI:	MVI	A,0	;mark include inactive
	STA	IFLAG
	LXI	H,SBUFF	;set DMA
	CALL	SDA
	LXI	D,SFCB	;restore source file position
	CALL	RRR
	LXI	H,SSIZE	;mark source buffer empty
	SHLD	SFBP
	LHLD	SFSBP	;reload the source buffer
	CALL	RSB
	CPI	''''
	CZ	GNC	;If end of filename string
;
	RET
;
;	RSB - Reload the source buffer.
;	Entry	HL = target source buffer pointer
;	Exit	 A = next source character
;
RSB:	CALL	GNC
	MOV	B,A
	XCHG
	LHLD	SFBP
	XCHG
	CALL	CHD
	JNZ	RSB	;loop to targer position
;
	MOV	A,B
	RET
;
;	GNC - Get next source file character.
;	Exit	 A = character
;
GNC:	LDA	IFLAG
	ORA	A
	JNZ	GIC	;If include active
;
	PUSH B ! PUSH D ! PUSH H
	LHLD	SFBP	;buffer pointer
	LXI	D,SSIZE
	CALL	CHD
	JNZ	GNC2	;If data available
;
;	Reload the source buffer.
;
	CALL	SSF	;select ADISK
	LXI	H,0	;reset the source buffer pointer
	SHLD	SFBP
	LXI	D,SFCB	;mark this file position
	CALL	SRR
	MVI	B,NSFBF	;records to read
	LXI	H,SBUFF	;buffer fwa
;
;	Read next record.
;
GNC1:	PUSH B ! PUSH H
	CALL	SDA	;set DMA
	LXI	D,SFCB
	CALL	RSR
	POP H ! POP B
	MVI	C,128
	JNZ	GNC3	;If end of information
;
;	Advance buffer.
;
	MOV	A,C
	CALL	HLA	;advance DMA
	DCR	B	;advance record count
	JNZ	GNC1	;loop for B records
;
;	Get next character.
;
GNC2:	LXI	D,SBUFF
	LHLD	SFBP
	PUSH	H	;save current pointer
	INX	H	;advance buffer pointer
	SHLD	SFBP
	POP	H	;current buffer pointer
	DAD	D
	MOV	A,M	;get character
	POP H ! POP D ! POP B
	RET
;
;	Process end of information on source file.
;
GNC3:	MVI	M,eof	;pack record with eof marks
	INX	H
	DCR	C
	JNZ	GNC3	;loop to end of record
	JMP	GNC2
;
;	PNC - Put next print file character.
;	Entry	 A = character
;
PNC:	PUSH	B
	MOV	B,A	;save character
	LDA	PDISK
	CPI	'Z'-'A'
	JZ	PNC2	;If no print file
;
	CPI	'X'-'A'
	MOV	A,B
	JNZ	PNC1	;If not printing console
;
;	Send the print file to the console.
;
	CALL	CCO
	JMP	PNC2
;
;	Pack print file character.
;
PNC1:	PUSH D ! PUSH H
	CALL	PPC	;pack character
	POP H ! POP D
PNC2:	POP	B
	RET
;
;	PPC - Pack print character.
;	Entry	 A = character
;
PPC:	LHLD	PFBP	;print file buffer pointer
	XCHG
	LXI	H,PBUFF
	DAD	D
	MOV	M,A	;char to buffer
	XCHG
	INX	H	;advance buffer pointer
	SHLD	PFBP
	XCHG
	LXI	H,PSIZE
	CALL	CHD
	RNZ		;If buffer not full
;
;	The print file buffer is full.
;
	CALL	SPF	;select PDISK
	CALL	SBA	;set DMA=RBUF
	LXI	H,0000	;reset buffer pointer
	SHLD	PFBP
	LXI	H,PBUFF
	LXI	D,PFCB	;print file fcb
	MVI	B,NPFBF	;record count
;
;	WBF - Write buffer to disk.
;	Entry	DE = .fcb
;		HL = buffer fwa
;		 B = record count
;
WBF:	MOV	A,M	;get character
	CPI	eof
	RZ		;If end of file
;
	PUSH B ! PUSH D
	MVI	C,128	;byte count
	LXI	D,RBUF	;move record to RBUF
	CALL	MMC
	POP D ! PUSH D	;get & save .fcb
	PUSH	H	;save buffer location
	CALL	WSR
	POP	H
	POP D ! POP B
	JNZ	WBF1	;If disk is full
;
	DCR	B	;advance record
	RZ		;If buffer dumped
;
	JMP	WBF	;loop for B records
;
;	Process full disk.
;
WBF1:	LXI	D,WBFA	;"disk is full"
	CALL	MSG
	JMP	EOR4	;go close files
;
WBFA:	DB	cr,lf,'Disk is full.',cr,lf,0
;
;	PNB - Put next object code byte.
;	Entry	 A = byte
;
PNB:	PUSH B ! PUSH D ! PUSH H
	CALL	PNB1
	POP H ! POP D ! POP B
	RET
;
;	Put next hex byte.
;	Entry	 A = byte
;
PNB1:	LHLD	HFBP	;hex file buffer pointer
	XCHG
	LXI	H,HBUFF
	DAD	D
	MOV	M,A	;pack byte
	XCHG
	INX	H	;advance buffer pointer
	SHLD	HFBP
	XCHG
	LXI	H,HSIZE
	CALL	CHD
	RNZ		;If buffer not full
;
;	The object code file buffer is full.
;
	CALL	SHF	;select HDISK
	CALL	SBA	;set DMA=RBUF
	LXI	H,0000	;reset hex buffer pointer
	SHLD	HFBP
	LXI	H,HBUFF
	LXI	D,HFCB	;hex file fcb
	MVI	B,NHFBF	;record count
	JMP	WBF	;go dump the buffer to disk
;
;	WPC - Write print file character.
;	Echo to console, if error.
;	Entry	 A = character
;
WPC:	MOV	C,A	;save char
	CALL	PNC	;put char to file
	LDA	PLBFWA	;examine column 1
	CPI	' '
	RZ		;If not an error line
;
;	Current line has been error flagged.
;
	LDA	PDISK
	CPI	'X'-'A'
	RZ		;If listing to console
;
	MOV	A,C	;send the line to console
	CALL	CCO
	RET
;
;	WPL - Write print line to print file.
;
WPL:	LDA	PLBFBP	;character count
	LXI	H,PLBFWA	;buffer fwa
WPL1:	ORA	A
	JZ	WPL2	;If done
;
	MOV	B,A	;save count
	MOV	A,M	;send char to file
	CALL	WPC
	INX	H	;advance index
	MOV	A,B	;advance char count
	DCR	A
	JMP	WPL1	;loop over print line
;
;	Process end of print line.
;
WPL2:	STA	PLBFBP	;reset line buffer pointer
	CALL	PEL	;pack cr,lf
	CALL	BFB	;blank fill the print line buffer
	RET
;
;	PER - Pack error flag in column 1.
;	Entry	 A = error flag as Ascii character
;
PER:	MOV	B,A	;save flag
	LXI	H,PLBFWA
	MOV	A,M
	CPI	' '
	RNZ		;If line already flagged
;
	MOV	M,B
	RET
;
;	PCF - Process chained source file.
;	Entry	 A = NEXTC
;		 C = LASTC
;	Exit	 Z = false, if not chaining, with
;		 A = NEXTC
;		 Z = true, if chained to new source file
;
PCF:	CPI	'%'
	RNZ		;If not chaining
;
	MOV	B,A	;save NEXTC
	MOV	A,C	;examine LASTC
	CPI	lf
	MOV	A,B
	RNZ		;If not 1st char of line, not chaining
;
;	Process chain directive.
;	Copy filename into source fcb.
;
	LXI	H,SFCB+1
	MVI	B,8
PCF1:	CALL	GNC	;get next char
	CPI	'.'
	JZ	PCF2	;If end of filename
;
	MOV	M,A
	INX	H
	DCR	B
	JZ	PCF3	;If 8 char moved
;
	JMP	PCF1	;loop over filename
;
;	Blank fill short filename.
;
PCF2:	MVI	M,' '
	INX	H
	DCR	B
	JNZ	PCF2	;loop to end of filename
;
;	Process filetype.
;
PCF3:	MVI	B,3
PCF4:	CALL	GNC	;get next char
	CPI '.' ! JZ PCF4	;If ".", ignore it
	CPI cr  ! JZ PCF5	;If chain filename error
;
	MOV	M,A
	INX	H
	DCR	B
	JNZ	PCF4	;loop over filetype
;
	LDA	HEXRCL	;save hex buffer
	PUSH	PSW
	CALL	RIF	;open new source file
	POP	PSW
	STA	HEXRCL
	XRA	A	;indicate chained to new file
	RET
;
;	Process chain filename error.
;
PCF5:	MVI	M,cr ! INX H
	MVI	M,00
	LXI	D,PCFA	;"chain error:"
	CALL	MSG
	LXI	D,SFCB+1	;display the name
	CALL	MSG
	JMP	EOR
;
PCFA:	DB	'Chain filename error.  Incomplete filetype: ',0
;
;	BHW - Buffer pack hex word.
;	Entry	HL = word
;		DE = pack location
;	Exit	DE = next pack location
;
BHW:	PUSH	H
	MOV	A,H	;pack high byte
	CALL	BHB
	POP	H	;pack low byte
	MOV	A,L
;
;	BHB - Buffer pack hex byte.
;	Entry	 A = byte
;		DE = pack location
;	Exit	DE = next pack location
;
BHB:	PUSH	PSW
	RRC ! RRC ! RRC ! RRC
	CALL	BHD	;pack high digit
	POP	PSW	;pack low digit
;
;	BHD - Buffer pack hex digit.
;	Entry	 A, low 4 bits = digit
;		DE = pack location
;	Exit	DE = next pack location
;
BHD:	ANI	0FH
	ADI	90H
	DAA
	ACI	40H
	DAA
;
;	BAC - Buffer pack Ascii character.
;	Entry	 A = char
;		DE = pack location
;	Exit	DE = next pack location
;
BAC:	PUSH	H
	STAX	D
	INX	D
	LXI	H,PLBFBP	;advance buffer char count
	INR	M
	POP	H
	RET
;
;	E2S - Emit two spaces.
;	E1S - Emit one space.
;
E2S:	CALL	E1S
E1S:	MVI	A,' '
	CALL	BAC
	RET
;
;	C2C - Convert two BCD digits to Ascii.
;	Entry	HL = .BCD byte
;		DE = pack location
;	Exit	DE = next pack location
;
C2C:	MOV	A,M	;get BCD
	PUSH	PSW
	RRC ! RRC ! RRC ! RRC
	CALL	C2C1	;pack high digit
	POP	PSW	;pack low digit
;
C2C1:	ANI	0FH
	ADI	'0'
	STAX	D
	INX	D
	RET
;
;	GDA - Get date & time string address.
;	Exit	HL = .date BCD string
;
GDA:	CALL	RCI	;DE=.common data area
	LXI	H,000AH
	DAD	D
	RET
;
;	PTM - Pack time of day in Ascii.
;	Entry	DE = pack location
;
PTM:	PUSH	D
	CALL	GDA	;HL=.date
	INX H ! INX H ! INX H
	POP	D
	CALL	PTM1	;pack hour
	CALL	PTM1	;pack minute
	CALL	C2C	;pack second
	RET
;
PTM1:	CALL	C2C	;pack 2 char
	MVI	A,':'
	STAX	D
	INX	D
	INX	H
	RET
;
;	PDT - Pack date in Ascii.
;	Entry	DE = pack location
;
PDT:	PUSH	D
	CALL	GDA	;HL=.date
	POP	D
	CALL	PDT1	;pack year
	CALL	PDT1	;pack month
	CALL	C2C	;pack day
	RET
;
PDT1:	CALL	C2C	;pack 2 char
	MVI	A,'/'
	STAX	D
	INX	D
	INX	H
	RET
;
;	PPL - Pack print line.
;	Entry	DE = .text line
;	line ends in 00
;
PPL:	XCHG
	LXI	D,PLBFWA	;preset pack location
	XRA	A		;preset buffer char count
	STA	PLBFBP
;
PPL1:	MOV	A,M
	ORA	A
	RZ			;If end of text
;
	CALL	BAC		;pack char
	INX	H
	JMP	PPL1
;
;	EST - Emit symbol text.
;	Entry	HL = symbol entry fwa
;	Exit	HL = next symbol entry fwa
;
XRFFPC	EQU	6	;cross ref first print column
EST:	INX H ! INX H	;skip collision field
	MOV	A,M	;get type and length
	ANI	0FH	;A=length
	INR	A
	LXI	D,PLBFWA+XRFFPC	;preset pack location
	MOV	C,A
	MVI	B,16+2
	MVI	A,XRFFPC
	STA	PLBFBP
;
;	Pack symbol printname.
;
EST1:	INX	H	;pack next char
	MOV	A,M
	CALL	BAC
	DCR	B
	DCR	C
	JNZ	EST1	;loop over printname
;
;	Space fill printname space in buffer.
;
EST2:	CALL	E1S	;one space
	DCR	B
	JNZ	EST2	;loop to end of field
;
;	Pack symbol value.
;
	PUSH	D	;save pack location
	INX H ! MOV E,M	;get symbol value
	INX H ! MOV D,M
	INX H ! XTHL	;HL=pack loc, stack=.symord
	XCHG		;HL=sym value, DE=pack loc
	CALL	BHW	;pack hex value
	CALL	E2S	;4 spaces
	CALL	E2S
	POP	H	;HL=.symord
	PUSH	D	;save pack location
	MOV E,M ! INX H	;get symbol table ordinal
	MOV D,M ! INX H
	XTHL		;HL=pack loc, stack=.next symbol
	XCHG		;HL=symord, DE=pack loc
	SHLD	XRFRRN	;set cross ref record number
	POP	H
	RET
;
;	ESR - Emit symbol references.
;	Entry	HL = .xref record
;		 C = reference count
;		DE = pack location
;
ESR:	INX H ! INX H
	INX H ! INX H	;step over symbol table entry address
	DCR	C
ESR1:	MVI	B,0
ESR2:	PUSH	D
	MOV E,M ! INX H
	MOV D,M ! INX H
	XTHL		;HL=pack loc, stack=rec loc
	XCHG		;HL=reference, DE=pack loc
	CALL	BHW
	CALL	E2S	;2 spaces
	POP	H	;.next reference
	DCR	C
	JZ	ESR3	;If end of references
;
;	Check print line.
;
	INR	B
	MOV	A,B
	CPI	8
	JNZ	ESR2	;If not full line
;
;	Emit this line.
;
	PUSH B ! PUSH H
	CALL	WPL	;write print line
	MVI	A,XRFFPC+18+8	;reset buffer char count
	STA	PLBFBP
	LXI	D,PLBFWA+XRFFPC+18+8
	POP H ! POP B
	JMP	ESR1
;
;	Emit last line.
;
ESR3:	CALL	WPL
	RET
;
;	LXR - List cross reference map.
;
LXR:	LDA	XFCB
	ORA	A
	RZ		;If no cross reference
;
	CALL	PEL	;cr,lf
	LXI	D,LXRB	;copy source filename
	LXI	H,SFCB+1
	MVI	C,8+3
	CALL	MMC
	LXI	D,LXRD	;pack date
	CALL	PDT
	LXI	D,LXRE	;pack time
	CALL	PTM
	LXI	D,LXRA	;emit heading
	CALL	PPL
	CALL	WPL
	CALL	PEL
;
;	Sort the cross reference table into symbol alphabetic order.
;
	LXI	D,XFCB		;close XREF file
	CALL	CLO
	CALL	TSA		;generate the symbol address table
	SHLD	TSLWA		;save table lwa+1
	MOV H,B ! MOV L,C	;save the symbol count
	SHLD	TSSIZ
	CALL	SXR		;sort the symbol address table
;
;	Emit symbols in symbol address table order.
;
	LXI	D,XFCB		;open XREF file
	CALL	OPN
	LHLD	TSSIZ		;set BC=symbol count
	MOV B,H ! MOV C,L
	LXI	H,0000		;reset table ordinal
LXR1:	PUSH B ! PUSH H
	CALL	GTA		;HL=address table address
	CALL	GSA		;HL=symbol table entry address
	CALL	GRN		;HL=XREF record number
	SHLD	XRFRRN
	LXI	H,XRFBUF	;set DMA
	CALL	SDA
	LXI	D,XFCB		;read the XREF record for this symbol
	CALL	RRR
	LHLD	XRFBUF+2	;get symbol table entry address
	CALL	EST		;emit symbol print name
;
;	DE = pack location.
;
	LXI	H,XRFBUF	;emit references to this symbol
	MOV	C,M
	CALL	ESR
;
;	Advance indexes.
;
	POP H ! INX H
	POP B ! DCX B
	MOV A,B ! ORA C
	JNZ	LXR1		;loop over sorted symbol address table
;
	RET
;
LXRA:	DB	' LISTXREF:   '
LXRB:	DB	'             '
LXRC:	DB	' Cross reference.  '
LXRD:	DB	'86/10/04.  '
LXRE:	DB	'10:22:33.  ',0
;
;	EOR - Process end of assembly.
;
EOR:	CALL	NPR	;check print file
	JZ	EOR2	;If no print file
;
;	List the cross reference map.
;
	CALL	LXR
;
;	Pack eof to end of buffer.
;
EOR1:	LHLD	PFBP
	MOV	A,L
	ORA	H
	JZ	EOR2	;If end of buffer
;
	MVI	A,eof
	CALL	PNC
	JMP	EOR1
;
EOR2:	LDA	HDISK
	CPI	'Z'-'A'
	JZ	EOR4	;If no hex file
;
;	Write the last hex record.
;
	LDA	HEXRCL	;check record length
	ORA	A
	CNZ	WHR
;
	LHLD	HEXPC	;write a zero length record
	SHLD	HEXBPC
	CALL	WHR
;
;	Fill hex buffer with eof.
;
EOR3:	LHLD	HFBP
	MOV	A,L
	ORA	H
	JZ	EOR4	;If end of buffer
;
	MVI	A,eof
	CALL	PNB
	JMP	EOR3
;
;	Close files.
;
EOR4:	CALL	NPR
	JZ	EOR5	;If no print file
;
	CALL	SPF	;select PDISK
	LXI	D,PFCB	;close print file
	CALL	CLO
	JZ	EOR8	;If close failed
;
EOR5:	LDA	HDISK
	CPI	'Z'-'A'
	JZ	EOR6	;If no hex file
;
	CALL	SHF	;select HDISK
	LXI	D,HFCB	;close hex file
	CALL	CLO
	JZ	EOR9	;If close failed
;
;	Close the cross reference file.
;
EOR6:	LXI	D,XFCB
	LDAX	D
	ORA	A
	CNZ	DEL	;If cross ref created
;
;	End of assembly.
;
	LXI	D,EORA	;"end of assembly"
EOR7:	CALL	MSG
	JMP	RWBT
;
;	Process print file close error.
;
EOR8:	LXI	D,EORB	;".PRN close error"
	CALL	MSG
	JMP	EOR5
;
;	Process hex file close error.
;
EOR9:	LXI	D,EORC	;".HEX close error"
	JMP	EOR7
;
EORA:	DB	cr,lf,'End of assembly.',cr,lf,0
EORB:	DB	cr,lf,'.PRN close error.',cr,lf,0
EORC:	DB	cr,lf,'.HEX close error.',cr,lf,0
;
;	PHB - Put hex byte.
;	Entry	 A = data byte
;
PHB:	PUSH B
	MOV	B,A
	LDA	HDISK
	CPI	'Z'-'A'
	MOV	A,B
	JZ	PHB4	;If no hex file
;
	PUSH D ! PUSH PSW
	LXI	H,HEXRCL	;current record length
	MOV	A,M
	ORA	A
	JZ	PHB2	;If starting a new record
;
;	Check record full.
;
	CPI	16
	JC	PHB1	;If space available
;
;	Dump the current record.
;
	CALL	WHR	;write hex record
	JMP	PHB2
;
;	Check sequential data byte.
;
PHB1:	LHLD	HEXPC	;DE=current hex fill PC
	XCHG
	LHLD	HEXBPC	;hex base PC
	MOV	C,A	;BC=current buffer loc
	MVI	B,0
	DAD	B
	CALL	CHD
	JZ	PHB3	;If sequential byte
;
;	Process non-sequential data byte.
;
	CALL	WHR	;dump the current record
;
;	Start a new record.
;
PHB2:	LHLD	HEXPC	;set the new base
	SHLD	HEXBPC
PHB3:	LXI	H,HEXRCL	;pack data byte
	MOV	E,M
	MVI	D,0
	INR	M	;advance record length
	LXI	H,HEXBUF
	DAD	D
	POP	PSW	;recover data byte
	MOV	M,A
	POP	D
PHB4:	POP	B
	RET
;
;	WHB - Write hex byte with checksum.
;	Entry	 A = character
;		 D = checksum
;	Exit	 D = checksum
;
WHB:	PUSH	PSW	;save char
	RRC ! RRC ! RRC ! RRC
	ANI	0FH
	CALL	WHD	;write hex digit
	POP PSW ! PUSH PSW
	ANI	0FH
	CALL	WHD	;write hex digit
	POP	PSW	;advance checksum
	ADD	D
	MOV	D,A
	RET
;
;	WHD - Write hex digit.
;	Entry	 A,low 4 bits = digit
;
WHD:	ADI	90H
	DAA
	ACI	40H
	DAA
	JMP	PNB
;
;	WHR - Write hex record.
;
WHR:	MVI	A,':'	;put preamble
	CALL	PNB
	LXI	H,HEXRCL	;.record length
	MOV	E,M
	XRA	A	;reset checksum
	MOV	D,A
	MOV	M,A	;reset next record length
	LHLD	HEXBPC
	MOV	A,E	;put record length
	CALL	WHB
	MOV	A,H	;put record address
	CALL	WHB
	MOV	A,L
	CALL	WHB
	LDA	HRTYPE	;put record type
	CALL	WHB
	MOV	A,E
	ORA	A
	JZ	WHR2	;If writing zero length record
;
;	Put the data bytes.
;
	LXI	H,HEXBUF
WHR1:	MOV	A,M
	INX	H
	CALL	WHB
	DCR	E	;advance byte count
	JNZ	WHR1	;loop over data bytes
;
;	Process hex record checksum.
;	Entry	 D = checksum
;
WHR2:	XRA	A	;negate checksum
	SUB	D
	CALL	WHB	;put checksum
;
;	Pack cr,lf.
;
	MVI	A,cr
	CALL	PNB
	MVI	A,lf
	CALL	PNB
	RET
;
ENDIOM	EQU	($ AND 0FF00H) + 100H
